/*
 * This file is part of Remote Support Desktop
 * https://gitlab.das-netzwerkteam.de/RemoteWebApp/remote-support-desktop
 * Copyright 2020 Daniel Teichmann <daniel.teichmann@das-netzwerkteam.de>
 * Copyright 2020 Mike Gabriel <mike.gabriel@das-netzwerktea.de>
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

import QtQuick 2.9
import QtQuick.Window 2.2
import QtQuick.Extras 1.4
import QtQuick.Controls 2.0
import QtQuick.Dialogs 1.2
import QtQuick.Controls.Material 2.3

ApplicationWindow {
    property int normal_width: 700
    property int normal_height: 400

    ToastManager {
        id: toast
    }

    width: normal_width
    height: normal_height

    // Make window not resizeable
    maximumWidth: normal_width * 1.25
    maximumHeight: normal_height * 1.25

    // Make window not resizeable
    minimumWidth: 500
    minimumHeight: 300

    id: window
    visible: true
    title: qsTr("Remote Support for your Desktop")

    onClosing: {
        mainqmladaptor.onCloseHandler();
    }

    function minimizeWindow() {
        showMinimized();
        console.log("Miniming window now...");
    }

    function showWindow() {
        showNormal();
        console.log("Opening window now...");
    }

    Button {
        id: start_support_button
        objectName: "start_support_button"
        text: qsTr("Start remote support session")
        anchors.bottom: parent.bottom
        anchors.bottomMargin: 15
        anchors.right: parent.right
        anchors.rightMargin: 15
        checkable: true

        onClicked: mainqmladaptor.handleConnectButtonClick(checked);
    }

    MessageDialog {
        id: message_dialog
        objectName: "message_dialog"
        title: qsTr("Remote Support for your Desktop")
        text: qsTr("You are not supposed to see this message.\nThis is a bug.")
        icon: StandardIcon.Critical
    }

    Connections {
        target: mainqmladaptor
        onShowToastSignal: {
            toast.show(text, durationMs)
        }
    }

    Connections {
        target: mainqmladaptor
        onShowMessageDialogChanged: {
            message_dialog.visible = show
        }
    }

    Connections {
        target: mainqmladaptor
        onMessageDialogTextChanged: {
            message_dialog.text = text
        }
    }

    Connections {
        target: mainqmladaptor
        onMessageDialogTitleChanged: {
            message_dialog.title = title
        }
    }

    Connections {
        target: mainqmladaptor
        onMessageDialogIconChanged: {
            message_dialog.icon = iconindex
        }
    }

    Label {
        id: explain_function_label
        y: 115
        width: window.width * 0.4
        height: window.height/2
        text: qsTr("Please tell your partner your PIN and the Support URL to connect to this computer")
        font.family: "Verdana"
        font.pointSize: 12
        visible: true
        enabled: false
        anchors.left: parent.left
        anchors.leftMargin: 15
        wrapMode: Text.WrapAtWordBoundaryOrAnywhere
        anchors.verticalCenterOffset: 0
        anchors.verticalCenter: parent.verticalCenter
    }

    Label {
        id: dbus_api_status_text
        y: 350
        width: parent.width / 2
        height: 22
        text: "Unknown state of Service"
        wrapMode: Text.WrapAtWordBoundaryOrAnywhere
        verticalAlignment: Text.AlignVCenter
        font.pointSize: 11
        fontSizeMode: Text.Fit
        anchors.leftMargin: 15 + dbus_api_status_indicator.width + 5
        objectName: "dbus_api_status_text"
        anchors.bottom: parent.bottom
        anchors.bottomMargin: 15
        anchors.left: parent.left

        StatusIndicator {
            id: dbus_api_status_indicator
            width: height
            height: parent.height
            objectName: "dbus_api_status_indicator"
            color: "#73d216"
            anchors.verticalCenter: parent.verticalCenter
            anchors.right: parent.left
            anchors.rightMargin: 5
            active: false
        }
    }

    Item {
        id: url_group
        objectName: "url_group"
        x: 301
        y: 154
        width: parent.width / 2
        height: parent.height * 0.25
        anchors.verticalCenterOffset: -height/2
        anchors.verticalCenter: parent.verticalCenter
        anchors.right: parent.right
        anchors.rightMargin: 0

        Button {
            id: copy_url_to_clipboard_button
            x: 460
            y: 2
            width: copy_url_to_clipboard_image.width + 6
            height: copy_url_to_clipboard_image.height + 6 + 10
            anchors.verticalCenter: url_text.verticalCenter
            flat: true
            display: AbstractButton.IconOnly
            anchors.leftMargin: 5
            highlighted: false
            anchors.left: url_text.right

            Image {
                id: copy_url_to_clipboard_image
                x: -230
                y: -2
                opacity: 0.65
                fillMode: Image.PreserveAspectFit
                anchors.verticalCenter: parent.verticalCenter
                anchors.horizontalCenter: parent.horizontalCenter
                //visible: false
                source: "images/into-clipboard.svg"
            }

            onClicked: {
                mainqmladaptor.handleCopyToClipboardButtonClick(url_text.text);
                toast.show(qsTr("Copied URL into clipboard!"), "1000");
            }

            ToolTip.text: qsTr("Copy the URL into the clipboard")
            hoverEnabled: true

            ToolTip.delay: 1000
            ToolTip.timeout: 5000
            ToolTip.visible: hovered
        }

        Label {
            id: url_text
            y: 0
            width: parent.width - copy_url_to_clipboard_button.width - 5
            height: parent.height/2
            text: mainqmladaptor.url
            verticalAlignment: Text.AlignVCenter
            horizontalAlignment: Text.AlignRight
            anchors.bottom: parent.bottom
            anchors.bottomMargin: 0
            anchors.left: parent.left
            anchors.leftMargin: 0
            font.bold: true
            font.pointSize: 13
            fontSizeMode: Text.Fit
            wrapMode: Text.WrapAtWordBoundaryOrAnywhere
        }

        Label {
            id: your_url_text
            y: 0
            width: parent.width
            height: parent.height/2
            text: qsTr("Remote Support URL")
            font.pointSize: 14
            anchors.left: parent.left
            anchors.leftMargin: 0
            fontSizeMode: Text.Fit
            anchors.top: parent.top
            anchors.topMargin: 0
            verticalAlignment: Text.AlignBottom
            anchors.bottomMargin: 5
            clip: false
            horizontalAlignment: Text.AlignLeft
        }
    }

    Item {
        id: pin_group
        x: 379
        y: 183
        width: parent.width/2
        objectName: "pin_group"
        height: parent.height * 0.25
        anchors.right: parent.right
        anchors.rightMargin: 0
        anchors.verticalCenterOffset: height/2
        anchors.verticalCenter: parent.verticalCenter

        Button {
            id: copy_pin_to_clipboard_button
            x: 460
            y: 97
            width: copy_pin_to_clipboard_image.width + 6
            height: copy_pin_to_clipboard_image.height + 6 + 10
            anchors.verticalCenterOffset: 0
            anchors.verticalCenter: pin_text.verticalCenter
            flat: true
            display: AbstractButton.IconOnly
            anchors.left: pin_text.right
            anchors.leftMargin: 5

            Image {
                id: copy_pin_to_clipboard_image
                x: -230
                y: -2
                opacity: 0.65
                anchors.verticalCenter: parent.verticalCenter
                anchors.horizontalCenter: parent.horizontalCenter
                source: "images/into-clipboard.svg"
                fillMode: Image.PreserveAspectFit
            }

            onClicked: {
                mainqmladaptor.handleCopyToClipboardButtonClick(pin_text.text);
                toast.show(qsTr("Copied PIN into clipboard!"), "1000");
            }

            ToolTip.text: qsTr("Copy the pin into the clipboard")
            hoverEnabled: true

            ToolTip.delay: 1000
            ToolTip.timeout: 5000
            ToolTip.visible: hovered
        }

        Label {
            objectName: "pin_text"
            id: pin_text
            width: parent.width - 5 - copy_pin_to_clipboard_button.width
            height: parent.height/2
            text: mainqmladaptor.pin
            font.bold: true
            font.pointSize: 20
            anchors.left: parent.left
            anchors.leftMargin: 0
            anchors.top: your_pin_text.bottom
            anchors.topMargin: 0
            font.letterSpacing: 10
            horizontalAlignment: Text.AlignRight
            verticalAlignment: Text.AlignVCenter
            fontSizeMode: Text.VerticalFit
        }

        Label {
            id: your_pin_text
            x: 15
            width: parent.width
            height: parent.height/2
            text: qsTr("Access pin to this computer")
            font.pointSize: 14
            anchors.top: parent.top
            anchors.topMargin: 0
            anchors.left: parent.left
            anchors.leftMargin: 0
            verticalAlignment: Text.AlignBottom
            horizontalAlignment: Text.AlignLeft
            clip: false
            fontSizeMode: Text.Fit
        }
    }

    Material.theme: theme.position < 1 ? Material.Light : Material.Dark

    Control {
        id: top_menu_bar_frame
        width: parent.width
        height: parent.height * 0.10

        background: Rectangle {
            color: parent.Material.background
            border.color: parent.Material.background
        }
        Material.background: "#0d5eaf"
        Material.foreground: "#ffffff"

        anchors.left: parent.left
        anchors.leftMargin: 0
        anchors.top: parent.top
        anchors.topMargin: 0

        Switch {
            id: theme
            height: parent.height
            anchors.margins: 10
            text: "Dark theme"
            anchors.left: sidemenu_open_button.right
            anchors.leftMargin: 15
            anchors.verticalCenter: parent.verticalCenter
            checked: true
        }

        Label {
            id: header_text
            height: parent.height
            color: "#ffffff"
            text: qsTr("Allow Remote Control")
            padding: 5
            font.family: "Verdana"
            font.pointSize: 20
            fontSizeMode: Text.Fit
            verticalAlignment: Text.AlignVCenter
            anchors.right: parent.right
            anchors.rightMargin: 5
            anchors.verticalCenter: parent.verticalCenter
            horizontalAlignment: Text.AlignRight
        }

        Button {
            id: sidemenu_open_button
            width: sidemenu_open_image.width
            height: parent.height + 10
            text: ""
            hoverEnabled: true
            display: AbstractButton.IconOnly
            anchors.left: parent.left
            anchors.leftMargin: 0
            anchors.top: parent.top
            anchors.topMargin: -5
            flat: true

            onClicked: page.visible = page.visible ? false : true, checked = false

            Image {
                id: sidemenu_open_image
                x: -5
                y: -4
                width: 40
                height: 40
                anchors.horizontalCenter: parent.horizontalCenter
                anchors.verticalCenter: parent.verticalCenter
                source: "images/menubar.png"
                fillMode: Image.PreserveAspectFit
            }
        }

    }

    Page {
        id: page
        x: 0
        y: 0
        width: 210
        height: 400
        visible: false

        Image {
            id: logo_image
            x: 640
            y: 17
            width: 50
            height: 50
            visible: true
            anchors.left: parent.left
            anchors.leftMargin: 14
            clip: false
            anchors.top: parent.top
            anchors.topMargin: 11
            source: "images/logo.png"
            fillMode: Image.PreserveAspectFit
        }

        Text {
            id: element2
            x: 14
            y: 85
            width: 180
            height: 27
            text: qsTr("Remote Control")
            fontSizeMode: Text.FixedSize
            verticalAlignment: Text.AlignVCenter
            horizontalAlignment: Text.AlignLeft
        }

        Text {
            id: element3
            x: 14
            y: 118
            width: 180
            height: 27
            text: qsTr("Remote View")
            font.pointSize: 12
            verticalAlignment: Text.AlignVCenter
            fontSizeMode: Text.FixedSize
            horizontalAlignment: Text.AlignLeft
        }

        Text {
            id: element4
            x: 14
            y: 151
            width: 181
            height: 31
            text: qsTr("Settings")
            fontSizeMode: Text.FixedSize
            verticalAlignment: Text.AlignVCenter
            font.pointSize: 12
        }

        Button {
            id: button1
            x: 171
            y: 0
            width: 40
            height: 40
            text: qsTr("x")
            checkable: true
            flat: true

            onClicked: page.visible = page.visible ? false : true, checked = false
        }
        }
        }






























/*##^## Designer {
    D{i:1;anchors_x:0;anchors_y:339}D{i:5;anchors_x:0}D{i:7;anchors_x:15;anchors_y:115}
D{i:11;anchors_height:20;anchors_width:20}D{i:10;anchors_y:154}D{i:14;anchors_x:47;anchors_y:0}
D{i:15;anchors_height:50}D{i:16;anchors_height:35;anchors_width:350;anchors_x:379;anchors_y:183}
D{i:19;anchors_height:35;anchors_x:73;anchors_y:35}D{i:18;anchors_height:35;anchors_x:73;anchors_y:35}
D{i:20;anchors_y:0}D{i:21;anchors_x:0;anchors_y:0}D{i:23;anchors_x:15;anchors_y:17}
D{i:24;anchors_height:20;anchors_width:200;anchors_x:15;anchors_y:10}
}
 ##^##*/
