/*
 * This file is part of Remote Support Desktop
 * https://gitlab.das-netzwerkteam.de/RemoteWebApp/remote-support-desktop
 * Copyright 2020 Daniel Teichmann <daniel.teichmann@das-netzwerkteam.de>
 * Copyright 2020 Mike Gabriel <mike.gabriel@das-netzwerktea.de>
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <QApplication>
#include <QQmlApplicationEngine>
#include <QQmlComponent>
#include <QQmlProperty>
#include <QQuickItem>
#include <QObject>
#include <QTranslator>
#include <QDebug>
#include <QQmlContext>
#include <QQuickStyle>
#include <signal.h>

#include "RWADBusAdaptor.cpp"
#include "session.h"

int main(int argc, char *argv[]) {
    QCoreApplication::setAttribute(Qt::AA_EnableHighDpiScaling);
    QApplication app(argc, argv);

    QQuickStyle::setStyle("Material");

    QTranslator translator;
    qDebug() << "Loading locale: qrc:/locales/bin/" + QLocale::system().name();
    if(translator.load(":/locales/bin/" + QLocale::system().name())) {
        app.installTranslator(&translator);
        qDebug() << "Loaded: " + QLocale::system().name() + " locale!";
    } else {
        qDebug() << "Unable to load translation";
    }

    QQmlApplicationEngine engine(&app);

    QScopedPointer<MainQMLAdaptor> main_gui (new MainQMLAdaptor(&app, &engine));
    //MainQMLAdaptor *main_gui = new MainQMLAdaptor(&app, &engine);
    // Make mainqmladaptor available to QML
    engine.rootContext()->setContextProperty("mainqmladaptor", main_gui.data());
    qDebug() << "'mainqmladaptor' is now available in QML context.";

    engine.load(QUrl(QStringLiteral("qrc:/main.qml")));
    if (engine.rootObjects().isEmpty())
        return -1;

    QScopedPointer<Session> session (new Session(&app, main_gui.data()));
    //Session *session = new Session(&app, &engine, main_gui);

    QObject::connect(main_gui.data(),
                         SIGNAL(minimizeWindow()),
                         engine.rootObjects().takeFirst(),
                         SLOT(minimizeWindow()));

    QObject::connect(main_gui.data(),
                         SIGNAL(showWindow()),
                         engine.rootObjects().takeFirst(),
                         SLOT(showWindow()));

    // We don't want users to have multiple instances of this app running
    QString tmpDirPath = QDir::tempPath() + "/rwa";
    QString tmpFilePath = tmpDirPath + "/remote-support-desktop-application-prevent-multiple-instances.lock";
    QDir tmpDir(tmpDirPath);
    if (!tmpDir.exists()) {
        // Ensure that the path exists
        tmpDir.mkpath(".");
    }
    QLockFile lockFile(tmpFilePath);
    qDebug() << "Checking for a lockfile at: " + tmpFilePath;

    if(!lockFile.tryLock(100)){
        qDebug() << QObject::tr("You already have this app running.");
        qDebug() << QObject::tr("Only one instance is allowed.");
        qDebug() << QObject::tr("Closing application now with an error.");

        return 1;
    }

    return app.exec();
}
