/*
 * This file is part of Remote Support Desktop
 * https://gitlab.das-netzwerkteam.de/RemoteWebApp/remote-support-desktop
 * Copyright 2020 Daniel Teichmann <daniel.teichmann@das-netzwerkteam.de>
 * Copyright 2020 Mike Gabriel <mike.gabriel@das-netzwerktea.de>
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#pragma once

#include <QObject>
#include <QQmlApplicationEngine>
#include <QQuickItem>
#include <QTimerEvent>
#include <QTranslator>
#include <QtDBus/QtDBus>

#include "RWADBusAdaptor.h"
#include "main_qmladaptor.h"
//#include "session.cpp"

class Session : public QObject
{
    Q_OBJECT
    Q_PROPERTY(QString status READ getStatus NOTIFY statusChanged) // this makes status available as a QML property
    Q_PROPERTY(int     id     READ getId     NOTIFY idChanged    ) // this makes id     available as a QML property
    Q_PROPERTY(QString url    READ getURL    NOTIFY urlChanged   ) // this makes url    available as a QML property
    Q_PROPERTY(QString pin    READ getPin    NOTIFY pinChanged   ) // this makes pin    available as a QML property
public:
    explicit Session(QObject *parent, MainQMLAdaptor *main_gui = nullptr);

    QString getStatus();
    QString getURL();
    int getId();
    QString getPin();

    void setStatus(QString status);
    void setURL(QString url);
    void setId(int id);
    void setPin(QString pin);

    // Starts a VNC Session
    void start_request_dbus();
    // Refreshes a VNC Session's status
    void refresh_status_request_dbus(int pid);
    // Stop the Session
    void stop_request_dbus(int pid);
    // Gets a VNC Session's status
    void status_request_dbus(int pid);
    // Returns true if Session is somewhat usable (Running, Alive, etc..)
    bool isSessionAliveOrRunning(QString status);

protected:
    QString _status;
    void statusTimerEvent();
    void init_vars();

private:
    MainQMLAdaptor* _main_gui;
    int _id;
    QString _url;
    QString _pin;
    OrgArcticaProjectRWAInterface* _dbus_rwa;
    void _initDBus();

    bool _minimizedBefore = false;
    void minimizeWindow();

signals:
    void finished();
    void statusChanged(QString status);
    void idChanged(int id);
    void urlChanged(QString URL);
    void pinChanged(QString pin);

public slots:
    void handleConnectButtonClick(bool checked);

    void start_dbus_replied(QDBusPendingCallWatcher *call);
    void stop_dbus_replied(QDBusPendingCallWatcher *call);
    void status_dbus_replied(QDBusPendingCallWatcher *call);

    void onCloseHandler();
};
