/*
 * This file is part of Remote Support Desktop
 * https://gitlab.das-netzwerkteam.de/RemoteWebApp/remote-support-desktop
 * Copyright 2020, 2021 Daniel Teichmann <daniel.teichmann@das-netzwerkteam.de>
 * Copyright 2020, 2021 Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

import QtQuick 2.0
import QtQuick.Controls 2.0
import QtQuick.Controls.Material 2.3
import rwa.toast.type 1.0

/**
 * adapted from StackOverflow:
 * http://stackoverflow.com/questions/26879266/make-toast-in-android-by-qml
 * GitHub Gist: https://gist.github.com/jonmcclung/bae669101d17b103e94790341301c129
 * Adapted to work with dark/light theming
 */

/**
  * @brief An Android-like timed message text in
  * a box that self-destroys when finished if desired
  */
Control {

    /**
      * Public
      */

    /**
      * @brief Shows this Toast
      *
      * @param {string} text Text to show
      * @param {real} duration Duration to show in milliseconds, defaults to 3000
      * @param {enum} type Type of toast. Available is:
      *               ToastType.Standard, ToastType.Info, ToastType.Warning
      *               ToastType.Success, ToastType.Error
      */
    function show(text, duration, type) {
        message.text = text;

        // checks if parameter was passed
        if (typeof duration !== "undefined") {
            time = Math.max(duration, 2 * fadeTime);
        } else {
            time = defaultTime;
        }

        if (typeof type !== "undefined" ) {
            if (type === ToastType.ToastStandard) {
                selectedColor = "#dcdedc";
            } else if (type === ToastType.ToastInfo) {
                selectedColor = "#0d5eaf";
            } else if (type === ToastType.ToastSuccess) {
                selectedColor = "#0daf36";
            } else if (type === ToastType.ToastWarning) {
                selectedColor = "#efef2a";
            } else if (type === ToastType.ToastError) {
                selectedColor = "#ed1212";
            }
        } else {
            selectedColor = "#dcdedc";
        }

        animation.start();
    }

    // whether this Toast will self-destroy when it is finished
    property bool selfDestroying: false

    /**
      * Private
      */

    id: root

    property color selectedColor: "#dcdedc"
    readonly property real defaultTime: 3000
    property real time: defaultTime
    readonly property real fadeTime: 300

    property real margin: 10

    anchors {
        left: parent.left
        right: parent.right
        margins: margin
    }

    height: message.height + margin

    background: Rectangle {
        color: (Material.theme == Material.Dark) ? "#212121" : "#dcdedc"
        border.color: selectedColor
        border.width: 1.5
        radius: margin
    }

    opacity: 0

    Text {
        id: message
        color: (Material.theme == Material.Dark) ? "#f1f1f1" : "#010101"
        wrapMode: Text.Wrap
        horizontalAlignment: Text.AlignHCenter
        anchors {
            top: parent.top
            left: parent.left
            right: parent.right
            margins: margin / 2
        }
    }

    SequentialAnimation on opacity {
        id: animation
        running: false


        NumberAnimation {
            to: .9
            duration: fadeTime
        }

        PauseAnimation {
            duration: time - 2 * fadeTime
        }

        NumberAnimation {
            to: 0
            duration: fadeTime
        }

        onRunningChanged: {
            if (!running && selfDestroying) {
                root.destroy();
            }
        }
    }
}
