/*
 * This file is part of Remote Support Desktop
 * https://gitlab.das-netzwerkteam.de/RemoteWebApp/rwa.support.desktopapp
 * Copyright 2020, 2021 Daniel Teichmann <daniel.teichmann@das-netzwerkteam.de>
 * Copyright 2020, 2021 Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <QApplication>
#include <QQmlApplicationEngine>
#include <QQmlComponent>
#include <QQmlProperty>
#include <QQuickItem>
#include <QObject>
#include <QTranslator>
#include <QDebug>
#include <QQmlContext>
#include <QQuickStyle>
#include <signal.h>

#include "DBusAPI.h"
#include "RWADBusAdaptor.cpp"
#include "session.h"
#include "scenes/add_rwahost_wizard/add_rwahost_wizard.h"
#include "scenes/remote_control/remote_control_manager.h"
#include "RWAHostModel.h"
#include "RWAHost.h"

int main(int argc, char *argv[]) {
    qDebug() << "This app was built on: " << __DATE__ <<  __TIME__;

    // We don't want users to have multiple instances of this app running
    QString tmpDirPath = QDir::tempPath() + "/rwa.support.desktopapp";
    QString tmpFilePath = tmpDirPath + "/prevent-multiple-instances.lock";
    QDir tmpDir(tmpDirPath);
    if (!tmpDir.exists()) {
        // Ensure that the path exists
        tmpDir.mkpath(".");
    }
    QLockFile lockFile(tmpFilePath);
    qDebug().noquote() << QString("Checking for a lockfile at: '%0'").arg(tmpFilePath);

    if(!lockFile.tryLock(100)){
        qCritical().noquote() << "You already have this app running.\n"
                           << "Only one instance is allowed.\n"
                           << "Closing application now with an error.";

        return 1;
    }

    QCoreApplication::setAttribute(Qt::AA_EnableHighDpiScaling);
    QApplication app(argc, argv);

    QQuickStyle::setStyle("Material");

    QTranslator translator;
    qDebug().noquote() << QString("Locales: Loading locale: qrc:/locales/%0")
                          .arg(QLocale::system().name());
    if(translator.load(":/locales/" + QLocale::system().name())) {
        app.installTranslator(&translator);
        qDebug().noquote() << "Locales: Loaded: " + QLocale::system().name() + " locale!";
    } else {
        qWarning() << "Locales: Unable to load translation!";
    }

    QQmlApplicationEngine engine(&app);

    QScopedPointer<DBusAPI> dbus_api (new DBusAPI());

    // Make 'mainqmladaptor' available to QML
    QScopedPointer<MainQMLAdaptor> main_gui (
                new MainQMLAdaptor(&app, &engine, dbus_api.data())
                );

    engine.rootContext()->setContextProperty("mainqmladaptor", main_gui.data());

    QObject::connect(dbus_api.data(),
                     SIGNAL(serviceGetWebAppHostsResponse(QJsonDocument*)),
                     main_gui.data(),
                     SLOT(get_web_app_hosts_response(QJsonDocument*)));
    dbus_api.data()->get_web_app_hosts_request();


    engine.load(QUrl(QStringLiteral("qrc:/src/main.qml")));
    if (engine.rootObjects().isEmpty())
        return -1;

    QObject::connect(main_gui.data(),
                     SIGNAL(minimizeWindow()),
                     engine.rootObjects().takeFirst(),
                     SLOT(minimizeWindow()));

    QObject::connect(main_gui.data(),
                     SIGNAL(showWindow()),
                     engine.rootObjects().takeFirst(),
                     SLOT(showWindow()));

    QObject::connect(engine.rootObjects().takeFirst()->
                     findChild<QObject*>("sidebar_drawer"),
                     SIGNAL(rwaHostSelected(QString)),
                     main_gui.data(),
                     SLOT(onRwaHostSelected(QString)));

    // Make 'AddRWAHostWizard' available to QML
    QScopedPointer<AddRWAHostWizard> wizard (
                new AddRWAHostWizard(&app,
                                      main_gui.data(),
                                      dbus_api.data())
                );
    engine.rootContext()->
            setContextProperty("add_rwahost_wizard", wizard.data());

    // Make 'remote_control_manager' available to QML
    QScopedPointer<RemoteControlManager> remote_mngr (
                new RemoteControlManager(&engine,
                                         main_gui.data(),
                                         dbus_api.data())
                );
    engine.rootContext()->
            setContextProperty("remote_control_manager", remote_mngr.data());

    return app.exec();
}
