/*
 * This file is part of Remote Support Desktop
 * https://gitlab.das-netzwerkteam.de/RemoteWebApp/rwa.support.desktopapp
 * Copyright 2020, 2021 Daniel Teichmann <daniel.teichmann@das-netzwerkteam.de>
 * Copyright 2020, 2021 Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#pragma once

#include <QQmlApplicationEngine>
#include <QQuickItem>
#include <QMessageBox>
#include <QApplication>
#include <QClipboard>
#include <QJsonObject>
#include <QJsonDocument>
#include <QJsonArray>
#include <QTimer>

#include "RWAHost.h"
#include "DBusAPI.h"

namespace Toast {
    Q_NAMESPACE
    enum ToastType {
        ToastStandard,
        ToastInfo,
        ToastSuccess,
        ToastWarning,
        ToastError
    };
    Q_ENUM_NS(ToastType)
}

/*#include <QObject>
namespace MyNamespace
{
    Q_NAMESPACE         // required for meta object creation
    enum EnStyle {
        STYLE_RADIAL,
        STYLE_ENVELOPE,
        STYLE_FILLED
    };
    Q_ENUM_NS(EnStyle)  // register the enum in meta object data
}*/

class MainQMLAdaptor : public QObject {
    Q_OBJECT
    // this makes rwaHostModel available as a QML property
    Q_PROPERTY(QList<QObject*> rwaHostModel READ getRWAHostModel NOTIFY rwaHostModelChanged)
    // this makes showMessageDialog available as a QML property
    Q_PROPERTY(bool showMessageDialog READ getShowMessageDialog NOTIFY showMessageDialogChanged)
    // this makes showMessageDialogTitle available as a QML property
    Q_PROPERTY(QString _messageDialogTitle READ getMessageDialogTitle NOTIFY messageDialogTitleChanged)
    // this makes showMessageDialogText available as a QML property
    Q_PROPERTY(QString _messageDialogText READ getMessageDialogText NOTIFY messageDialogTextChanged)
    // this makes showMessageDialogIcon available as a QML property
    Q_PROPERTY(QMessageBox::Icon _messageDialogIcon READ getMessageDialogIcon NOTIFY messageDialogIconChanged)


public:
    explicit MainQMLAdaptor(QObject *parent, QQmlApplicationEngine *engine = nullptr,
                            DBusAPI *dbus_api = nullptr);

    void setRWAHostModel(QList<QObject*> *rwa_hosts);

    void main_content_push(QString);
    void main_content_pop(QString);
    void main_content_replace(QString);

    bool openMessageDialog(QString title, QString text, QMessageBox::Icon);
    QString getMessageDialogTitle();
    QString getMessageDialogText();
    QMessageBox::Icon getMessageDialogIcon();
    bool getShowMessageDialog();

signals:
    void showMessageDialogChanged(bool show);
    void messageDialogTextChanged(QString text);
    void messageDialogTitleChanged(QString title);
    void messageDialogIconChanged(int iconindex);

    void minimizeWindow();
    void showWindow();

    void rwaHostModelChanged(QList<QObject*>);

    void onCloseSignal();

    void showToastSignal(QString text, QString durationMs, int type);

protected:
    DBusAPI *_dbus_api;
    QList<QObject*> *_rwaHostModel;

private:
    QQmlApplicationEngine *_engine;
    RWAHost *_selected_rwa_host;

    bool _showMessageDialog;
    QString _messageDialogTitle;
    QString _messageDialogText;
    QMessageBox::Icon _messageDialogIcon;

public slots:
    void get_web_app_hosts_response(QJsonDocument *doc);

    void addRWAHost(RWAHost *rwa_host);
    void removeRWAHost(RWAHost *rwa_host);
    void setRWAHostSelected(bool value);

    // No pointer because QML doesn't
    // like this type much with pointer
    QList<QObject*> getRWAHostModel();

    RWAHost *getSelectedRWAHost();

    void onRwaHostSelected(QString host_uuid);
    void onCloseHandler();

    // arg type is actually Toast::ToastType
    void showToast(QString text,
                   uint durationMs = 3000,
                   uint type = 0);
};
