/*
 * This file is part of Remote Support Desktop
 * https://gitlab.das-netzwerkteam.de/RemoteWebApp/rwa.support.desktopapp
 * Copyright 2021 Daniel Teichmann <daniel.teichmann@das-netzwerkteam.de>
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include "add_rwahost_wizard.h"
#include "../../RWADBusAdaptor.h"
#include "../../RWAHost.h"

AddRWAHostWizard::AddRWAHostWizard(QObject *parent,
                                     MainQMLAdaptor *main_gui, DBusAPI *dbus_api) : QObject(parent) {
    Q_ASSERT(main_gui != nullptr);
    Q_ASSERT(dbus_api != nullptr);

    _dbus_api = dbus_api;
    _main_gui = main_gui;

    // _dbus_api --serviceAddWebAppHostResponse-> this.add_web_app_host_response()
    QObject::connect(_dbus_api,
                     SIGNAL(serviceAddWebAppHostResponse(QJsonDocument*)),
                     this,
                     SLOT(add_web_app_host_response(QJsonDocument*)));
}

void AddRWAHostWizard::processStep1(QString host_url, QString host_alias) {
    qDebug() << "Processing Step 1 with args: " << host_url << host_alias;

    if(host_alias == "" || host_url == "") {
        QString reason = tr("Both textfields can't be empty!");
        emit step1Failed(reason, Toast::ToastType::ToastWarning);
        qDebug().noquote() << reason;
        return;
    }

    return add_server(host_url, host_alias);
}

void AddRWAHostWizard::processStep2() {
    qDebug() << "Processing Step 2 with args: No Args.";
    emit step2Failed(tr("The features you expected here are not yet implemented."), Toast::ToastType::ToastWarning);
    // Just show placeholder scene now.
    emit step2Success();
}

void AddRWAHostWizard::add_server(QString host_url, QString host_alias) {
    _dbus_api->add_web_app_host_request(host_url, host_alias);
}

void AddRWAHostWizard::add_web_app_host_response(QJsonDocument *doc) {
    // Q_ASSERT lets the program crash immediatly at startup,
    // when the session service is not started.
    // Don't use Q_ASSERT(doc != nullptr); instead use:
    if (doc == nullptr) {
        _main_gui->setRWAHostSelected(false);
        _main_gui->showToast(tr("Can't connect to underlying session service!"),
                             9800,
                             Toast::ToastType::ToastError);
        return;
    }

    // Get the QJsonObject
    QJsonObject jObject = doc->object();
    QVariantMap mainMap = jObject.toVariantMap();

    // Status of request
    QString request_status = mainMap["status"].toString();
    if (request_status == "success") {
        _dbus_api->get_web_app_hosts_request();

        qInfo() << "Successfully added a new RWAHost.";
        emit step1Success();
    } else {
        qCritical().noquote() << tr("An error occured while adding a new host!");

        uint toast_type = Toast::ToastType::ToastStandard;
        QString reason = tr("An error occured while adding a new host!");
        QString type = mainMap["type"].toString();

        if(type == "connection"){
            reason = tr("Couldn't connect to the specified host!");
            toast_type = Toast::ToastType::ToastError;
            qCritical().noquote() << reason;
        } else if (type == "duplicate") {
            reason = tr("The specified host was already added!");
            toast_type = Toast::ToastType::ToastWarning;
            qCritical().noquote() << reason;
        } else if (type == "invalid_url") {
            reason = tr("The specified host address is not valid!");
            toast_type = Toast::ToastType::ToastWarning;
            qCritical().noquote() << reason;
        } else if (type == "permission_denied") {
            reason = tr("The specified host address does not grant access!");
            toast_type = Toast::ToastType::ToastError;
            qCritical().noquote() << reason;
        } else if (type == "unsupported_server") {
            reason = tr("The specified host address is not supported!");
            toast_type = Toast::ToastType::ToastError;
            qCritical().noquote() << reason;
        }
        emit step1Failed(reason, toast_type);

        return;
    }
}
